<?php
namespace EkobyteCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Repeater;
use \Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Ekobyte Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Ekobyte_Process extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'process';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Process Step', 'ekobytecore' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'ekobyte-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'ekobytecore' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'ekobytecore' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

        /**
         * Layout section
         */
        $this->start_controls_section(
            'ekobyte_layout',
            [
                'label' => esc_html__('Design Layout', 'ekobytecore'),
            ]
        );
        $this->add_control(
            'ekobyte_design_style',
            [
                'label' => esc_html__('Select Layout', 'ekobytecore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__('Layout 1', 'ekobytecore'),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->end_controls_section();

        
        /**
         * Title and content section
         */
        $this->start_controls_section(
            'ekobyte_section_title',
            [
                'label' => esc_html__('Title & Content', 'ekobytecore'),
            ]
        );

        $this->add_control(
            'ekobyte_section_title_show',
            [
                'label' => esc_html__( 'Section Title & Content', 'ekobytecore' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'ekobytecore' ),
                'label_off' => esc_html__( 'Hide', 'ekobytecore' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );
        
        $this->add_control(
            'ekobyte_sub_title',
            [
                'label' => esc_html__('Sub Title', 'ekobytecore'),
                'description' => ekobyte_get_allowed_html_desc( 'basic' ),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Ekobyte Sub Title', 'ekobytecore'),
                'placeholder' => esc_html__('Type Sub Heading Text', 'ekobytecore'),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'ekobyte_sub_title_color',
            [
                'label' => __( 'Sub Title Color', 'ekobytecore' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .te-section-title .short-title' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_control(
            'ekobyte_title',
            [
                'label' => esc_html__('Title', 'ekobytecore'),
                'description' => ekobyte_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Ekobyte Title Here', 'ekobytecore'),
                'placeholder' => esc_html__('Type Heading Text', 'ekobytecore'),
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'ekobyte_title_color',
            [
                'label' => __( 'Title Color', 'ekobytecore' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .te-section-title .title' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'ekobyte_title_tag',
            [
                'label' => esc_html__('Title HTML Tag', 'ekobytecore'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'h1' => [
                        'title' => esc_html__('H1', 'ekobytecore'),
                        'icon' => 'eicon-editor-h1'
                    ],
                    'h2' => [
                        'title' => esc_html__('H2', 'ekobytecore'),
                        'icon' => 'eicon-editor-h2'
                    ],
                    'h3' => [
                        'title' => esc_html__('H3', 'ekobytecore'),
                        'icon' => 'eicon-editor-h3'
                    ],
                    'h4' => [
                        'title' => esc_html__('H4', 'ekobytecore'),
                        'icon' => 'eicon-editor-h4'
                    ],
                    'h5' => [
                        'title' => esc_html__('H5', 'ekobytecore'),
                        'icon' => 'eicon-editor-h5'
                    ],
                    'h6' => [
                        'title' => esc_html__('H6', 'ekobytecore'),
                        'icon' => 'eicon-editor-h6'
                    ]
                ],
                'default' => 'h2',
                'toggle' => false,
            ]
        );

        $this->add_responsive_control(
            'ekobyte_align',
            [
                'label' => esc_html__('Alignment', 'ekobytecore'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'text-left' => [
                        'title' => esc_html__('Left', 'ekobytecore'),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'text-center' => [
                        'title' => esc_html__('Center', 'ekobytecore'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'text-right' => [
                        'title' => esc_html__('Right', 'ekobytecore'),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'default' => 'left',
                'toggle' => false,
            ]
        );
        $this->end_controls_section();

        
        /**
         * Process list
         */
        $this->start_controls_section(
            'ekobyte_process',
            [
                'label' => esc_html__('Process List', 'ekobytecore'),
                'description' => esc_html__( 'Control all the style settings from Style tab', 'ekobytecore' ),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new \Elementor\Repeater();
        $repeater->add_control(
            'ekobyte_process_bg_color',
            [
                'label' => esc_html__('Background Color', 'ekobytecore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} {{CURRENT_ITEM}} .te-process-step' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        // step image
        $repeater->add_control(
            'ekobyte_step_image',
            [
                'label' => esc_html__('Step Image', 'ekobytecore'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );

        // process image/icon
        $repeater->add_control(
            'ekobyte_process_icon_type',
            [
                'label' => esc_html__('Process Icon/Image', 'ekobytecore'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'image',
                'options' => [
                    'image' => esc_html__('Image', 'ekobytecore'),
                    'icon' => esc_html__('Icon', 'ekobytecore'),
                ],
            ]
        );

        $repeater->add_control(
            'ekobyte_icon_image',
            [
                'label' => esc_html__('Upload Image', 'ekobytecore'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'condition' => [
                    'ekobyte_process_icon_type' => 'image'
                ]
            ]
        );

        if (ekobyte_is_elementor_version('<', '2.6.0')) {
            $repeater->add_control(
                'icon',
                [
                    'show_label' => false,
                    'type' => Controls_Manager::ICON,
                    'label_block' => true,
                    'default' => 'fa fa-star',
                    'condition' => [
                        'ekobyte_process_icon_type' => 'icon'
                    ]
                ]
            );
        } else {
            $repeater->add_control(
                'selected_icon',
                [
                    'show_label' => false,
                    'type' => Controls_Manager::ICONS,
                    'fa4compatibility' => 'icon',
                    'label_block' => true,
                    'default' => [
                        'value' => 'far fa-star',
                        'library' => 'regular',
                    ],
                    'condition' => [
                        'ekobyte_process_icon_type' => 'icon'
                    ]
                ]
            );
        }

        // process step
        $repeater->add_control(
            'ekobyte_process_step', [
                'label' => esc_html__('Process Step', 'ekobytecore'),
                'description' => ekobyte_get_allowed_html_desc( 'basic' ),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => esc_html__('Step - 01', 'ekobytecore'),
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'ekobyte_process_step_color',
            [
                'label' => __( 'Step Color', 'ekobytecore' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-simple-step.v2 .process-card .process-card-header h4' => 'color: {{VALUE}}',
                ],
            ]
        );

        $repeater->add_control(
            'ekobyte_process_title', [
                'label' => esc_html__('Process Title', 'ekobytecore'),
                'description' => ekobyte_get_allowed_html_desc( 'basic' ),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => esc_html__('Process title here', 'ekobytecore'),
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'ekobyte_process_title_color',
            [
                'label' => __( 'Process Title Color', 'ekobytecore' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .process-card-header h4' => 'color: {{VALUE}}',
                ],
            ]
        );

        $repeater->add_control(
            'ekobyte_process_description',
            [
                'label' => esc_html__('Description', 'ekobytecore'),
                'description' => ekobyte_get_allowed_html_desc( 'intermediate' ),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'default' => 'There are many variations of passages of Lorem Ipsum available, but the majority have suffered.',
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'ekobyte_process_description_color',
            [
                'label' => __( 'Description Color', 'ekobytecore' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .process-card p' => 'color: {{VALUE}}',
                ],
            ]
        );

        

        $this->add_control(
            'ekobyte_process_list',
            [
                'label' => esc_html__('Process - List', 'ekobytecore'),
                'type' => \Elementor\Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'ekobyte_process_title' => esc_html__('Discover', 'ekobytecore'),
                    ],
                    [
                        'ekobyte_process_title' => esc_html__('Define', 'ekobytecore')
                    ],
                    [
                        'ekobyte_process_title' => esc_html__('Develop', 'ekobytecore')
                    ]
                ],
                'title_field' => '{{{ ekobyte_process_title }}}',
            ]
        );
        $this->add_responsive_control(
            'ekobyte_process_align',
            [
                'label' => esc_html__( 'Alignment', 'ekobytecore' ),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'text-left' => [
                        'title' => esc_html__( 'Left', 'ekobytecore' ),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'text-center' => [
                        'title' => esc_html__( 'Center', 'ekobytecore' ),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'text-right' => [
                        'title' => esc_html__( 'Right', 'ekobytecore' ),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'toggle' => true,
                'separator' => 'before',
            ]
        );
        $this->end_controls_section();

        
        /**
         * Style section
         */
		$this->start_controls_section(
			'section_style',
			[
				'label' => __( 'Style', 'ekobytecore' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'text_transform',
			[
				'label' => __( 'Text Transform', 'ekobytecore' ),
				'type' => Controls_Manager::SELECT,
				'default' => '',
				'options' => [
					'' => __( 'None', 'ekobytecore' ),
					'uppercase' => __( 'UPPERCASE', 'ekobytecore' ),
					'lowercase' => __( 'lowercase', 'ekobytecore' ),
					'capitalize' => __( 'Capitalize', 'ekobytecore' ),
				],
				'selectors' => [
					'{{WRAPPER}} .title' => 'text-transform: {{VALUE}};',
				],
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render the widget ouekobyteut on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
        $settings = $this->get_settings_for_display();
		?>

		<?php
			$this->add_render_attribute('title_args', 'class', 'title');
        ?>	
            <!-- Process Step Area Start -->
            <div class="te-process-step-area style-1">
                <div class="container">
                    <div class="row">
                        <?php if ( !empty($settings['ekobyte_section_title_show']) ) : ?>
                        <div class="col-12">
                            <div class="te-section-title justify-content-center text-center">
                                <div class="te-section-content">
                                    <?php if ( !empty($settings['ekobyte_sub_title']) ) : ?>
                                        <div>
                                            <span class="short-title"><?php echo ekobyte_kses( $settings['ekobyte_sub_title'] ); ?></span>
                                        </div>
                                    <?php endif; ?>
                                    <?php
                                        if ( !empty($settings['ekobyte_title' ]) ) :
                                            printf( '<%1$s %2$s>%3$s</%1$s>',
                                                tag_escape( $settings['ekobyte_title_tag'] ),
                                                $this->get_render_attribute_string( 'title_args' ),
                                                ekobyte_kses( $settings['ekobyte_title' ] )
                                            );
                                    endif;?>
                                </div>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>
                    <div class="row">
                        <?php foreach ($settings['ekobyte_process_list'] as $key => $item) : ?> 
                            <!-- Single Service Start !-->
                            <div class="col-12 col-md-6 col-lg-4 elementor-repeater-item-<?php echo $item['_id']; ?>">
                                <div class="te-process-step">
                                    <div class="te-content-wrapper">
                                        <div class="te-counter-wrapper">
                                            <?php if (!empty($item['ekobyte_process_step' ])): ?>
                                                <span class="counter-number"><?php echo ekobyte_kses($item['ekobyte_process_step' ]); ?></span>
                                            <?php endif; ?>
                                            <div class="shape">
                                                <?php if (!empty($item['ekobyte_step_image']['url'])): ?>  
                                                    <img src="<?php echo $item['ekobyte_step_image']['url']; ?>" alt="<?php echo get_post_meta(attachment_url_to_postid($item['ekobyte_step_image']['url']), '_wp_attachment_image_alt', true); ?>"> 
                                                <?php endif; ?> 
                                            </div>
                                        </div>
                                        <div class="te-title-wrapper">
                                            <div class="icon">
                                                <?php if($item['ekobyte_process_icon_type'] !== 'image') : ?>
                                                    <?php if (!empty($item['icon']) || !empty($item['selected_icon']['value'])) : ?>
                                                        <?php ekobyte_render_icon($item, 'icon', 'selected_icon'); ?>
                                                    <?php endif; ?>   
                                                    <?php else : ?>                                
                                                    <?php if (!empty($item['ekobyte_icon_image']['url'])): ?>  
                                                        <img src="<?php echo $item['ekobyte_icon_image']['url']; ?>" alt="<?php echo get_post_meta(attachment_url_to_postid($item['ekobyte_icon_image']['url']), '_wp_attachment_image_alt', true); ?>">
                                                    <?php endif; ?> 
                                                <?php endif; ?>
                                            </div>
                                            <?php if (!empty($item['ekobyte_process_title' ])): ?>
                                                <h3 class="title"><?php echo ekobyte_kses($item['ekobyte_process_title' ]); ?></h4>
                                            <?php endif; ?>
                                        </div>
                                        <div class="content">
                                            <?php if (!empty($item['ekobyte_process_description' ])): ?>
                                                <p class="desc"><?php echo ekobyte_kses($item['ekobyte_process_description']); ?></p>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <!-- Single Service End !-->
                        <?php endforeach; ?>   
                    </div>
                </div>
            </div>
            <!-- Process Step Area End -->
        <?php 
	}
}

$widgets_manager->register( new Ekobyte_Process() );