<?php
namespace EkobyteCore;

use EkobyteCore\PageSettings\Page_Settings;
use Elementor\Controls_Manager;


/**
 * Class Plugin
 *
 * Main Plugin class
 * @since 1.2.0
 */
class Ekobyte_Core_Plugin {

	/**
	 * Instance
	 *
	 * @since 1.2.0
	 * @access private
	 * @static
	 *
	 * @var Plugin The single instance of the class.
	 */
	private static $_instance = null;

	/**
	 * Instance
	 *
	 * Ensures only one instance of the class is loaded or can be loaded.
	 *
	 * @since 1.2.0
	 * @access public
	 *
	 * @return Plugin An instance of the class.
	 */
	public static function instance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;
	}

	/**
	 * Add Category
	 */

    public function ekobyte_core_elementor_category($manager)
    {

        $categories = [];
		$categories['ekobytecore'] =
			[
				'title' => __( 'Element Helper ( Ekobyte )', 'ekobytecore' ),
				'icon'  => 'eicon-banner'
			];

		$old_categories = $manager->get_categories();
		$categories = array_merge($categories, $old_categories);

		$set_categories = function ( $categories ) {
			$this->categories = $categories;
		};

		$set_categories->call( $manager, $categories );
		
    }
		
		

	/**
	 * widget_scripts
	 *
	 * Load required plugin core files.
	 *
	 * @since 1.2.0
	 * @access public
	 */
	public function widget_scripts() {
		wp_register_script( 'ekobytecore', plugins_url( '/assets/js/hello-world.js', __FILE__ ), [ 'jquery' ], false, true );
	}

	/**
	 * Editor scripts
	 *
	 * Enqueue plugin javascripts integrations for Elementor editor.
	 *
	 * @since 1.2.1
	 * @access public
	 */
	public function editor_scripts() {
		add_filter( 'script_loader_tag', [ $this, 'editor_scripts_as_a_module' ], 10, 2 );

		wp_enqueue_script(
			'ekobytecore-editor',
			plugins_url( '/assets/js/editor/editor.js', __FILE__ ),
			[
				'elementor-editor',
			],
			'1.2.1',
			true
		);
	}


	/**
	 * ekobyte_enqueue_editor_scripts
	 */
    function ekobyte_enqueue_editor_scripts()
    {
        wp_enqueue_style('ekobyte-element-addons-editor', EKOBYTE_ADDONS_URL . 'assets/css/editor.css', null, '1.0');
    }

    

	/**
	 * Force load editor script as a module
	 *
	 * @since 1.2.1
	 *
	 * @param string $tag
	 * @param string $handle
	 *
	 * @return string
	 */
	public function editor_scripts_as_a_module( $tag, $handle ) {
		if ( 'ekobytecore-editor' === $handle ) {
			$tag = str_replace( '<script', '<script type="module"', $tag );
		}

		return $tag;
	}

	/**
	 * Register Widgets
	 *
	 * Register new Elementor widgets.
	 *
	 * @since 1.2.0
	 * @access public
	 *
	 * @param Widgets_Manager $widgets_manager Elementor widgets manager.
	 */
	public function register_widgets( $widgets_manager ) {
		// Its is now safe to include Widgets files
		foreach($this->ekobytecore_widget_list() as $widget_file_name){
			require_once( EKOBYTE_ELEMENTS_PATH . "/{$widget_file_name}.php" );
		}
	}

	public function ekobytecore_widget_list() {
		return [
			'heading',
			'ekobyte-btn',
			'hero-banner',
			'about',
			'services',
			'team',
			'process',
			'ekobyte-testimonial',
			'blog-post',
			'contact-form',
			'work-gallery-project',
			'company-benefit',
			'ekobyte-faq',
			'ekobyte-counter',
			'team-details',
			'service-details',
			'project-details',
			'home-2-cta',
			'home-1-newsletter',
			'home-2-newsletter',
			'about-talk-to-us',
		];
	}

	/**
	 * Add page settings controls
	 *
	 * Register new settings for a document page settings.
	 *
	 * @since 1.2.1
	 * @access private
	 */
	private function add_page_settings_controls() {
		require_once( __DIR__ . '/page-settings/manager.php' );
		new Page_Settings();
	}


	/**
	 * Register controls
	 *
	 * @param Controls_Manager $controls_Manager
	 */

    public function register_controls(Controls_Manager $controls_Manager)
    {
        include_once(EKOBYTE_ADDONS_DIR . '/controls/ekobytegradient.php');
        $ekobytegradient = 'EkobyteCore\Elementor\Controls\Group_Control_EkobyteGradient';
        $controls_Manager->add_group_control($ekobytegradient::get_type(), new $ekobytegradient());

        include_once(EKOBYTE_ADDONS_DIR . '/controls/ekobytebggradient.php');
        $ekobytebggradient = 'EkobyteCore\Elementor\Controls\Group_Control_EkobyteBGGradient';
        $controls_Manager->add_group_control($ekobytebggradient::get_type(), new $ekobytebggradient());
    }


    

    public function ekobyte_add_custom_icons_tab($tabs = array()){

        // Feather icons
        $feather_icons = array(
            'feather-activity',
            'feather-airplay',
            'feather-alert-circle',
            'feather-alert-octagon',
            'feather-alert-triangle',
            'feather-align-center',
            'feather-align-justify',
            'feather-align-left',
            'feather-align-right',
        );

        $tabs['ekobyte-feather-icons'] = array(
            'name' => 'ekobyte-feather-icons',
            'label' => esc_html__('Ekobyte - Feather Icons', 'ekobytecore'),
            'labelIcon' => 'ekobyte-icon',
            'prefix' => '',
            'displayPrefix' => 'ekobyte',
            'url' => EKOBYTE_ADDONS_URL . 'assets/css/feather.css',
            'icons' => $feather_icons,
            'ver' => '1.0.0',
		); 

		// Custom icons
		$custom_icons = array(
			'icon-24-help',
			'icon-chack-mark-solid',
			'icon-angle-arrow-right',
			'icon-angle-arrow-left',
			'icon-arrow-down',
			'icon-arrow-left',
			'icon-arrow-right',
			'icon-arrow-top-right',
			'icon-call',
			'icon-envelope',
			'icon-envelope-solid',
			'icon-facebook',
			'icon-instagram',
			'icon-linkedin',
			'icon-twitter',
			'icon-pinterest',
			'icon-google-plus',
			'icon-location-pin',
			'icon-location-pin-solid',
			'icon-group-icon-3',
			'icon-group-icon-4',
			'icon-group-icon-5',
			'icon-group-icon-6',
			'icon-group-icon-7',
			'icon-group-icon-8',
			'icon-chack-mark',
			'icon-user-group',
			'icon-Outer-Lines',
			'icon-Outer-Lines-1',
			'icon-Outer-Lines-2',
			'icon-Outer-Lines-3',
			'icon-Outer-Lines-4',
			'icon-Outer-Lines-5',
			'icon-Outer-Lines-6',
			'icon-Outline',
			'icon-play',
			'icon-plus',
			'icon-folder',
			'icon-user',
			'icon-user-solid',
			'icon-phone-book',
			'icon-dobble-angle-left',
			'icon-dobble-angle-right',
			'icon-user-stars',
			'icon-quote-right',
			'icon-start',
			'icon-start-solid',
			'icon-paper-plan',
			'icon-paper-plan-solid',
			'icon-calendar',
			'icon-chats',
			'icon-location',
			'icon-search',
		  );
		
		$tabs['ekobyte-custom-icons'] = array(
			'name' => 'ekobyte-custom-icons',
			'label' => esc_html__('Ekobyte - Custom Icons', 'ekobytecore'),
			'labelIcon' => 'ekobyte-icon',
			'prefix' => '',
			'displayPrefix' => 'ekobyte',
			'url' => EKOBYTE_ADDONS_URL . 'assets/css/ekobyte-customicon.css',
			'icons' => $custom_icons,
			'ver' => '1.0.0',
		); 

		// Font Awesome Pro
        $fontawesome_icons = array(
	        'angle-up',
	        'check',
	        'times',
	        'calendar',
	        'language',
	        'shopping-cart',
	        'bars',
	        'search',
	        'map-marker',
	        'arrow-right',
	        'arrow-left',
	        'arrow-up',
	        'arrow-down',
	        'angle-right',
	        'angle-left',
	        'angle-up',
	        'angle-down',
	        'phone',
	        'users',
	        'user',
	        'map-marked-alt',
	        'trophy-alt',
	        'envelope',
	        'marker',
	        'globe',
	        'broom',
	        'home',
	        'bed',
	        'chair',
	        'bath',
	        'tree',
	        'laptop-code',
	        'cube',
	        'cog',
	        'play',
	        'trophy-alt',
	        'heart',
	        'truck',
	        'user-circle',
	        'map-marker-alt',
	        'comments',
	         'award',
	        'bell',
	        'book-alt',
	        'book-open',
	        'book-reader',
	        'graduation-cap',
	        'laptop-code',
	        'music',
	        'ruler-triangle',
	        'user-graduate',
	        'microscope',
	        'glasses-alt',
	        'theater-masks',
	        'atom'
        );

        $tabs['ekobyte-fontawesome-icons'] = array(
            'name' => 'ekobyte-fontawesome-icons',
            'label' => esc_html__('Ekobyte - Fontawesome Pro Light', 'ekobytecore'),
            'labelIcon' => 'ekobyte-icon',
            'prefix' => 'fa-',
            'displayPrefix' => 'fal',
            'url' => EKOBYTE_ADDONS_URL . 'assets/css/fontawesome-all.min.css',
            'icons' => $fontawesome_icons,
            'ver' => '1.0.0',
        );        

        return $tabs;
    }


	/**
	 *  Plugin class constructor
	 *
	 * Register plugin action hooks and filters
	 *
	 * @since 1.2.0
	 * @access public
	 */
	public function __construct() {

		// Register widget scripts
		add_action( 'elementor/frontend/after_register_scripts', [ $this, 'widget_scripts' ] );

		// Register widgets
		add_action( 'elementor/widgets/register', [ $this, 'register_widgets' ] );

		// Register editor scripts
		add_action( 'elementor/editor/after_enqueue_scripts', [ $this, 'editor_scripts' ] );

		add_action('elementor/elements/categories_registered', [$this, 'ekobyte_core_elementor_category']);

		// Register custom controls
	    add_action('elementor/controls/controls_registered', [$this, 'register_controls']);

	    add_filter('elementor/icons_manager/additional_tabs', [$this, 'ekobyte_add_custom_icons_tab']);

	    // $this->ekobyte_add_custom_icons_tab();

	    add_action('elementor/editor/after_enqueue_scripts', [$this, 'ekobyte_enqueue_editor_scripts'] );

		$this->add_page_settings_controls();

	}


}

// Instantiate Plugin Class
Ekobyte_Core_Plugin::instance();